/* ***************************************************************** 
    MESQUITE -- The Mesh Quality Improvement Toolkit

    Copyright 2004 Sandia Corporation and Argonne National
    Laboratory.  Under the terms of Contract DE-AC04-94AL85000 
    with Sandia Corporation, the U.S. Government retains certain 
    rights in this software.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License 
    (lgpl.txt) along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
    diachin2@llnl.gov, djmelan@sandia.gov, mbrewer@sandia.gov, 
    pknupp@sandia.gov, tleurent@mcs.anl.gov, tmunson@mcs.anl.gov      
   
  ***************************************************************** */
// -*- Mode : c++; tab-width: 3; c-tab-always-indent: t; indent-tabs-mode: nil; c-basic-offset: 3 -*-

/*! \file AngleQM.hpp

Header file for the Mesquite::AngleQM class

\author Shankar Prasad Sastry
\date   9th July 2008
 */

#ifndef AngleQM_hpp
#define AngleQM_hpp

#include <vector>
#include "Mesquite.hpp"
#include "MsqError.hpp"
#include "ElementQM.hpp"
#include "AveragingQM.hpp"
#include "Vector3D.hpp"
#include "Matrix3D.hpp"
#include "Exponent.hpp"
#include "Mesquite_all_headers.hpp"

using namespace std;

namespace MESQUITE_NS
{
   class MsqMeshEntity;
   class PatchData;
   class MsqError;

   /*! \class AngleQM
     \brief Computes the mean ratio quality metric
     of given element.
     
     The metric does not use the sample point functionality or the
     compute_weighted_jacobian.  It evaluates the metric at
     the element vertices, and uses the anisotropic ideal element.
     It does require a feasible region, and the metric needs
     to be minimiseds.
   */
   class AngleQM : public ElementQM, public AveragingQM
   {
   public:
 
     //MESQUITE_EXPORT AngleQM(MsqError &err, double pow=1.0);
     MESQUITE_EXPORT AngleQM()
	  : AveragingQM(QualityMetric::LINEAR)
	{
	}
     
     //! virtual destructor ensures use of polymorphism during destruction
     MESQUITE_EXPORT virtual ~AngleQM(){
	}
     //virtual ~AngleQM();

      //! 1 if metric should be minimized, -1 if metric should be maximized.
     virtual int get_negate_flag() const
	{ return -1; }
     virtual msq_std::string get_name() const
	{ return "Minimum Angle Quality Metric"; }

     virtual
     bool evaluate( PatchData& pd, 
                    size_t handle, 
                    double& value, 
                    MsqError& err )

{
	  vector<Vector3D> vert;
	  Vector3D cross, normal;
	  Vector3D temp_vec[6], temp_v;
	  double vol, fval[6], fval_temp[6], dummy;
	
	  const MsqMeshEntity* e = &pd.element_by_index(handle);
	  EntityTopology topo = e->get_element_type();

	  
	  switch(topo) {
	  case TRIANGLE:
		  if (value==1e6)
		  {
			bool ret;
			double mm = value;
			ret = multi_evaluate_tri(pd, handle, mm, err, tt);
			value=mm;
			return ret;
		  }
  	    	pd.get_element_vertex_coordinates(handle, vert, err);  MSQ_ERRZERO(err);
	       cross = (vert[1] - vert[0]) * (vert[2] - vert[0]);
		vol = cross.length();
	      if (pd.domain_set()) { // need domain to check for inverted elements
		pd.get_domain_normal_at_corner( handle, 0, normal, err ); MSQ_ERRZERO(err);
		if ((cross % normal) < -MSQ_MIN)
		  vol = -vol;
	      }

		value = asin(vol/((vert[1]-vert[0]).length() * (vert[2]-vert[0]).length()) );
		dummy = asin(vol/((vert[2]-vert[1]).length() * (vert[0]-vert[1]).length()) );
		if(dummy<value)
			value = dummy;
		dummy = asin(vol/((vert[0]-vert[2]).length() * (vert[1]-vert[2]).length()) );
		if(dummy<value)
			value = dummy;
		
	    if (value<tt)
		{
			//printf("here oh boy! %lf %lf\n",value, tt);
			return false;
		}
	    return true;
	    break;


	  case TETRAHEDRON:
		  if (value==1e6)
		  {
			bool ret;
			double mm = value;
			ret = multi_evaluate(pd, handle, mm, err, tt);
			value=mm;
			return ret;
		  }

  	    	pd.get_element_vertex_coordinates(handle, vert, err);  MSQ_ERRZERO(err);
	     	vol=(vert[1]-vert[0])%((vert[2]-vert[0])*(vert[3]-vert[0]));
		//vol = fabs(vol);
			//sum of edges squared


			temp_vec[0] = temp_v=vert[1]-vert[0];
			fval[0]=temp_v.length();
			temp_vec[1] = temp_v=vert[2]-vert[0];
			fval[1]=temp_v.length();
			temp_vec[2] = 	temp_v=vert[3]-vert[0];
			fval[2]=temp_v.length();
			temp_vec[3] = temp_v=vert[2]-vert[1];
			fval[3]=temp_v.length();
			temp_vec[4] = temp_v=vert[3]-vert[1];
			fval[4]=temp_v.length();
			temp_vec[5] = temp_v=vert[3]-vert[2];
			fval[5]=temp_v.length();

	value = atan (vol/(fval[0]*fval[1]*fval[2] + (temp_vec[0]%temp_vec[1])*fval[2] + (temp_vec[1]%temp_vec[2])*fval[0] + (temp_vec[2]%temp_vec[0])*fval[1]));
	dummy = atan (vol/(fval[0]*fval[3]*fval[4] + (temp_vec[0]%temp_vec[3])*fval[4] + (temp_vec[3]%temp_vec[4])*fval[0] + (temp_vec[4]%temp_vec[0])*fval[3]));
	if (dummy<value)
		value=dummy;
	dummy = atan (vol/(fval[1]*fval[3]*fval[5] + (temp_vec[1]%temp_vec[3])*fval[5] + (temp_vec[3]%temp_vec[5])*fval[1] + (temp_vec[5]%temp_vec[1])*fval[3]));
	if (dummy<value)
		value=dummy;
	dummy = atan (vol/(fval[2]*fval[4]*fval[5] + (temp_vec[2]%temp_vec[4])*fval[5] + (temp_vec[4]%temp_vec[5])*fval[2] + (temp_vec[5]%temp_vec[2])*fval[4]));
	if (dummy<value)
		value=dummy;



	//DIHEDRAL ANGLE

	/*Vector3D n1, n2;

	n1 = (vert[2]-vert[0])*(vert[1]-vert[0]);
	n2 = (vert[3]-vert[0])*(vert[2]-vert[0]);
	value = asin((n1*n2).length()/(n1.length()*n2.length()));


	n1 = (vert[1]-vert[0])*(vert[2]-vert[0]);
	n2 = (vert[3]-vert[0])*(vert[1]-vert[0]);
	value = asin((n1*n2).length()/(n1.length()*n2.length()));


	n1 = (vert[2]-vert[1])*(vert[3]-vert[1]);
	n2 = (vert[0]-vert[1])*(vert[2]-vert[1]);
	value = asin((n1*n2).length()/(n1.length()*n2.length()));

	n1 = (vert[2]-vert[0])*(vert[1]-vert[0]);
	n2 = (vert[3]-vert[0])*(vert[2]-vert[0]);
	value = asin((n1*n2).length()/(n1.length()*n2.length()));*/
		
	    /*if (value<tt)
		{
			//printf("here oh boy! %lf %lf\n",value, tt);
			return false;
		}*/
		value=vol;
	    return true;
	    break;
	    
	  case POLYGON:
	  case QUADRILATERAL:
	  case POLYHEDRON:
	  case HEXAHEDRON:
	  case PRISM:
	  case SEPTAHEDRON:
	  case MIXED:
	  default:
	    MSQ_SETERR(err)(MsqError::UNSUPPORTED_ELEMENT,
			    "Element type (%d) not supported in here!",(int)topo);
	    return false;
	  } // end switch over element type
	  return true;
	}





    private:
      // arrays used in Hessian computations 
      // We allocate them here, so that one allocation only is done.
      // This gives a big computation speed increase.
      Vector3D mCoords[4]; // Vertex coordinates for the (decomposed) elements
      Vector3D mGradients[32]; // Gradient of metric with respect to the coords
      Matrix3D mHessians[80]; // Hessian of metric with respect to the coords
      double   mMetrics[8]; // Metric values for the (decomposed) elements
      bool multi_evaluate( PatchData& pd, 
					     size_t handle, 
					     double& m, 
					     MsqError& err,
					     double& tvalue)
{
		vector<Vector3D> vert;
	  	Vector3D temp_vec[6], tvec[6], temp_v;
	  	double vol, fval[6], tval[6], dummy[6];

		pd.get_element_vertex_coordinates(handle, vert, err);  MSQ_ERRZERO(err);
	     	vol=(vert[1]-vert[0])%((vert[2]-vert[0])*(vert[3]-vert[0]));
		
			temp_vec[0] = temp_v=vert[1]-vert[0];
			fval[0]=temp_v.length();
			temp_vec[1] = temp_v=vert[2]-vert[0];
			fval[1]=temp_v.length();
			temp_vec[2] = 	temp_v=vert[3]-vert[0];
			fval[2]=temp_v.length();
			temp_vec[3] = temp_v=vert[2]-vert[1];
			fval[3]=temp_v.length();
			temp_vec[4] = temp_v=vert[3]-vert[1];
			fval[4]=temp_v.length();
			temp_vec[5] = temp_v=vert[3]-vert[2];
			fval[5]=temp_v.length();

	dummy[0] = atan (vol/(fval[0]*fval[1]*fval[2] + (temp_vec[0]%temp_vec[1])*fval[2] + (temp_vec[1]%temp_vec[2])*fval[0] + (temp_vec[2]%temp_vec[0])*fval[1]));
	dummy[1] = atan (vol/(fval[0]*fval[3]*fval[4] + (temp_vec[0]%temp_vec[3])*fval[4] + (temp_vec[3]%temp_vec[4])*fval[0] + (temp_vec[4]%temp_vec[0])*fval[3]));
	dummy[2] = atan (vol/(fval[1]*fval[3]*fval[5] + (temp_vec[1]%temp_vec[3])*fval[5] + (temp_vec[3]%temp_vec[5])*fval[1] + (temp_vec[5]%temp_vec[1])*fval[3]));
	dummy[3] = atan (vol/(fval[2]*fval[4]*fval[5] + (temp_vec[2]%temp_vec[4])*fval[5] + (temp_vec[4]%temp_vec[5])*fval[2] + (temp_vec[5]%temp_vec[2])*fval[4]));
	m=0;	
	for(int i=0; i<4; i++)
	{
		if (dummy[i] < tvalue)
		{
			m = 1e10;
			//printf("here we go %lf %lf %lf\n", fval, vol, tvalue);
			return false;
		}
		m += log(dummy[i] - tvalue); 
	}


	return true;
	}


      bool multi_evaluate_tri( PatchData& pd, 
					     size_t handle, 
					     double& m, 
					     MsqError& err,
					     double& tvalue)
{
		vector<Vector3D> vert;
		Vector3D cross, normal;
	  	double vol, dummy[6];
  	    	pd.get_element_vertex_coordinates(handle, vert, err);  MSQ_ERRZERO(err);
	      cross = (vert[1] - vert[0]) * (vert[2] - vert[0]);
		vol = cross.length();
	      if (pd.domain_set()) { // need domain to check for inverted elements
		pd.get_domain_normal_at_corner( handle, 0, normal, err ); MSQ_ERRZERO(err);
		if ((cross % normal) < -MSQ_MIN)
		  vol = -vol;
	      }

		dummy[0] = asin(vol/((vert[1]-vert[0]).length() * (vert[2]-vert[0]).length()) );
		dummy[1] = asin(vol/((vert[2]-vert[1]).length() * (vert[0]-vert[1]).length()) );
		dummy[2] = asin(vol/((vert[0]-vert[2]).length() * (vert[1]-vert[2]).length()) );
		
	m=0;	
	for(int i=0; i<3; i++)
	{
		if (dummy[i] < tvalue)
		{
			m = 1e10;
			//printf("here we go %lf %lf %lf\n",dummy[i], vol, tvalue);
			return false;
		}
		m += log(dummy[i] - tvalue); 
	}


	return true;
	}
   };



} //namespace


#endif // AngleQM_hpp
